﻿
/****************************************************************************/
/*Copyright (c) 2011, Florent DEVILLE.                                      */
/*All rights reserved.                                                      */
/*                                                                          */
/*Redistribution and use in source and binary forms, with or without        */
/*modification, are permitted provided that the following conditions        */
/*are met:                                                                  */
/*                                                                          */
/* - Redistributions of source code must retain the above copyright         */
/*notice, this list of conditions and the following disclaimer.             */
/* - Redistributions in binary form must reproduce the above                */
/*copyright notice, this list of conditions and the following               */
/*disclaimer in the documentation and/or other materials provided           */
/*with the distribution.                                                    */
/* - The names of its contributors cannot be used to endorse or promote     */
/*products derived from this software without specific prior written        */
/*permission.                                                               */
/* - The source code cannot be used for commercial purposes without         */
/*its contributors' permission.                                             */
/*                                                                          */
/*THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       */
/*"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT         */
/*LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS         */
/*FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE            */
/*COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,       */
/*INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,      */
/*BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;          */
/*LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER          */
/*CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT        */
/*LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN         */
/*ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE           */
/*POSSIBILITY OF SUCH DAMAGE.                                               */
/****************************************************************************/

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using Microsoft.Xna.Framework;

namespace GE.World.Entities
{
    enum eSide
    {
        eSidePlayer,
        eSideEnemy,
        eSideNeutral
    }

    enum eEntityType
    { 
        eEntityPlayer,
        eEntityPlatform,
        eEntityLadder,
        eEntityEnemy,
        eNumberOfEntityType
    }

    enum ePhysicGroup
    {
        ePhysicPlayer =         0x000001,
        ePhysicPlatform =       0x000002,
        ePhysicEnemy =          0x000004,
        ePhysicBullet =         0x000008,
        ePhysicSpike =          0x000010,
        ePhysicLadder =         0x000020,
        //ePhysicSpike = 0x000040,
        ePhysicCount
    }

    /// <summary>
    /// Indicate a direction locked (used for the camera)
    /// </summary>
    enum eLockedDirection
    {
        eLockedHorizontal =         0x000001,   //01
        eLockedVertical =           0x000002,   //10
        eLockedBoth =               0x000003,   //11
        eLockedNone =               0x000000,   //00
        eLockedCount =              4
    }

    abstract class WorldEntity
    {
        #region Variables

        /// <summary>
        /// position of the entity
        /// </summary>
        protected Vector2 _v2Position;

        /// <summary>
        /// Previous position of the entity
        /// </summary>
        protected Vector2 _v2PreviousPosition;

        /// <summary>
        /// flag indicates if the entity is active or not. An active entity is updated
        /// </summary>
        protected bool _bActive;

        /// <summary>
        /// The number of point lost by an opponent when it collides with this entity
        /// </summary>
        protected int _iDamages;

        /// <summary>
        /// HP of the entity
        /// </summary>
        protected int _iHealthPoint;

        /// <summary>
        /// Side of the entity
        /// </summary>
        protected eSide _side;

        /// <summary>
        /// Type of entity
        /// </summary>
        protected eEntityType _entityType;

        #endregion

        #region Update & Render

        /// <summary>
        /// Update the entity
        /// </summary>
        virtual public void update() { }

        /// <summary>
        /// Render the entity
        /// </summary>
        virtual public void render() { }

        /// <summary>
        /// Activate the entity
        /// </summary>
        virtual public void activate() 
        {
            _bActive = true;
        }

        /// <summary>
        /// Initialise the entity. Load all the data which are not loaded during the level loading.
        /// </summary>
        virtual public void init() { }

        #endregion

        #region Properties

        /// <summary>
        /// Get or set the previous position
        /// </summary>
        public Vector2 PreviousPosition { get { return _v2PreviousPosition; } }

        /// <summary>
        /// Get or set the active state of the entity
        /// </summary>
        public virtual bool Active { get { return _bActive; } set { _bActive = value; } }

        /// <summary>
        /// Get the side of the entity
        /// </summary>
        public eSide Side { get { return _side; } }

        /// <summary>
        /// Get the damages the entity gives to an enemy
        /// </summary>
        public int Damages { get { return _iDamages; } set { _iDamages = value; } }

        /// <summary>
        /// Get the health of an entity
        /// </summary>
        public int HP { get { return _iHealthPoint; } set { _iHealthPoint = value; } }

        /// <summary>
        /// Get or set the position of the entity
        /// </summary>
        public Vector2 Position { get { return _v2Position; } set { _v2Position = value; } }

        /// <summary>
        /// Get the type of the entity
        /// </summary>
        public eEntityType EntityType { get { return _entityType; } }

        /// <summary>
        /// Compute and return the screen position of the entity
        /// </summary>
        public Vector2 ScreenPosition { get { return Position - World.Instance.CameraPosition; } }

        #endregion

        #region Tools

        /// <summary>
        /// Fonction applied when an entity dies
        /// </summary>
        virtual public void die() 
        {
            _bActive = false;
        }

        /// <summary>
        /// Fonction to apply damages to an entity
        /// </summary>
        /// <param name="damages"></param>
        virtual public void hurt(int damages) 
        {
            _iHealthPoint -= damages;
            if (_iHealthPoint <= 0) die();
        }

        #endregion
    }
}
